/** @file   AStarAlgorithm.cpp
 * @brief   Implementation of AStarAlgorithm class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/02/24 12:20:24 $
 * @author  Tomi Lamminsaari
 */

#include "AStarAlgorithm.h"
#include <vector>
#include <algorithm>
#include "eng2dPrivateConstants.h"
using namespace std;

namespace eng2d {

///
/// Constants, datatypes and static methods
/// ============================================================================




///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
AStarAlgorithm::AStarAlgorithm( MMapDataProvider& aMapProvider ) :
  iMapProvider( aMapProvider ),
  iMaxNodes( KAStarMaximumPathNodes )   // from eng2dPrivateConstants.h
{
}



/** Destructor.
 */
AStarAlgorithm::~AStarAlgorithm()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

bool AStarAlgorithm::FindPath( const Vec2D& aStart, const Vec2D& aGoal )
{
  vector<PathNode> openNodes;
  vector<PathNode> closedNodes;
  vector<PathNode> pathNodes;
  
  PathNode node;
  PathNode bestNode;
  bool pathFound = false;
  
  node.iPosition = aStart;
  node.iGone = 0;
  node.iHeuristic = iMapProvider.ApproximateDistance( aStart, aGoal );
  node.iTotalCost = node.iGone + node.iHeuristic;
  node.iParentPosition = Vec2D(0,0);
  openNodes.push_back( node );
  make_heap( openNodes.begin(), openNodes.end() );
  
  int counter1 = 0;
  while ( openNodes.empty() == false ) {
    sort_heap( openNodes.begin(), openNodes.end() );
    bestNode = openNodes.front();
    
    pop_heap( openNodes.begin(), openNodes.end() );
    openNodes.pop_back();
    closedNodes.push_back( bestNode );
    
    // If we've reached the goal position, we stop.
    if ( iMapProvider.IsSameNode( bestNode.iPosition, aGoal ) == true ) {
      // Route found.
      pathFound = true;
      break;
    }
    
    if ( closedNodes.size() > iMaxNodes ) {
      // Route would take too many nodes. Quit.
      pathFound = false;
      break;
    }
    
    // Create a vector for adjacent nodes.
    vector<Vec2D> adjacentNodes;
    iMapProvider.GetAdjacentNodes( adjacentNodes, bestNode.iPosition );
    
    for ( int i=0; i < adjacentNodes.size(); i++ ) {
      node.iPosition = adjacentNodes.at(i);
      node.iGone = bestNode.iGone + iMapProvider.GetTerrainFactor( node.iPosition );
      node.iHeuristic = iMapProvider.ApproximateDistance( node.iPosition, aGoal );
      
      node.iTotalCost = node.iGone + node.iHeuristic;
      node.iParentPosition = bestNode.iPosition;
      
      bool nodeFound = false;
      // Check if this node is already in array of OPEN
      for ( int j=0; j < openNodes.size(); j++ ) {
        if ( iMapProvider.IsSameNode( node.iPosition, openNodes.at(j).iPosition ) == true ) {
          if ( node.iGone < openNodes.at(j).iGone ) {
            openNodes.at(j).iGone = node.iGone;
            openNodes.at(j).iTotalCost = node.iGone + openNodes.at(j).iHeuristic;
            openNodes.at(j).iParentPosition = node.iParentPosition;
          }
          nodeFound = true;
          break;
        }
      }
      
      if ( nodeFound == false ) {
        // The node was not found from openNodes. Check if it's already in
        // closedNodes
        for ( int j=0; j < closedNodes.size(); j++ ) {
          if ( iMapProvider.IsSameNode(node.iPosition, closedNodes.at(j).iPosition ) == true ) {
            if ( node.iGone < closedNodes.at(j).iGone ) {
              closedNodes.at(j).iGone = node.iGone;
              closedNodes.at(j).iTotalCost = node.iGone + closedNodes.at(j).iHeuristic;
              closedNodes.at(j).iParentPosition = node.iParentPosition;
            }
            nodeFound = true;
            break;
          }
        }
      }
    
      if ( nodeFound == false ) {
        // The node was not found from openNodes nor closedNodes. Push new node
        // to open nodes.
        openNodes.push_back( node );
        push_heap( openNodes.begin(), openNodes.end() );
        make_heap( openNodes.begin(), openNodes.end() );
      }
    }
  }
  
  // Now we've either found the route or not.
  if ( closedNodes.size() > 0 ) {
    // Create the path from elements in closedNodes array
    pathNodes.clear();
    pathNodes.push_back( closedNodes.back() );
    closedNodes.pop_back();
    while ( closedNodes.empty() == false ) {
      if ( iMapProvider.IsSameNode( closedNodes.back().iPosition,
                                    pathNodes.back().iParentPosition ) == true ) {
        pathNodes.push_back( closedNodes.back() );
      }
      closedNodes.pop_back();
    }
    
    // Create the route points.
    for ( int i=(pathNodes.size()-1); i >= 0; i-- ) {
      iPath.push_back( pathNodes.at(i).iPosition );
    }
  }
  return pathFound;
}

const MPathFinderAlgo::NodeArray& AStarAlgorithm::GetPathNodes() const
{
  return iPath;
}

void AStarAlgorithm::ResetPath()
{
  iPath.clear();
}


///
/// New public methods
/// ============================================================================

void AStarAlgorithm::SetMaxRouteNodes( int aMaxNodes )
{
  iMaxNodes = aMaxNodes;
}



///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================


}  // end of namespace
